// Withdraw Page JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // Initialize withdrawal functionality
    initWithdrawPage();
});

function initWithdrawPage() {
    // Initialize method tabs
    initMethodTabs();
    
    // Initialize saved addresses
    initSavedAddresses();
    
    // Initialize withdrawal calculator
    initWithdrawalCalculator();
    
    // Initialize withdrawal form
    initWithdrawalForm();
    
    // Initialize table actions
    initTableActions();
    
    // Initialize modals
    initModals();
}

// Method Tabs Functionality
function initMethodTabs() {
    const tabs = document.querySelectorAll('.method-tab');
    const contents = document.querySelectorAll('.method-content');
    
    tabs.forEach(tab => {
        tab.addEventListener('click', function() {
            const method = this.dataset.method;
            
            // Remove active class from all tabs and contents
            tabs.forEach(t => t.classList.remove('active'));
            contents.forEach(c => c.classList.remove('active'));
            
            // Add active class to current tab
            this.classList.add('active');
            
            // Show corresponding content
            const content = document.getElementById(`${method}-method`);
            if (content) {
                content.classList.add('active');
            }
        });
    });
}

// Saved Addresses Functionality
function initSavedAddresses() {
    const savedAddresses = document.querySelectorAll('.saved-address input[type="radio"]');
    const walletAddressInput = document.getElementById('wallet-address');
    
    savedAddresses.forEach(address => {
        address.addEventListener('change', function() {
            if (this.checked) {
                walletAddressInput.value = this.value;
            }
        });
    });
    
    // Add address button
    const addAddressBtn = document.querySelector('.btn-add-address');
    if (addAddressBtn) {
        addAddressBtn.addEventListener('click', function() {
            showAddAddressModal();
        });
    }
}

// Withdrawal Calculator Functionality
function initWithdrawalCalculator() {
    const amountInput = document.getElementById('withdrawal-amount');
    const slider = document.getElementById('amount-slider');
    const amountActions = document.querySelectorAll('.amount-action');
    const withdrawableAmount = 3250; // This would come from server
    
    // Update slider and input sync
    if (slider && amountInput) {
        slider.addEventListener('input', function() {
            amountInput.value = this.value;
            updateFeeCalculation(this.value);
        });
        
        amountInput.addEventListener('input', function() {
            let value = parseFloat(this.value);
            if (isNaN(value)) value = 50;
            if (value < 50) value = 50;
            if (value > withdrawableAmount) value = withdrawableAmount;
            
            slider.value = value;
            this.value = value.toFixed(2);
            updateFeeCalculation(value);
        });
        
        amountInput.addEventListener('blur', function() {
            let value = parseFloat(this.value);
            if (isNaN(value)) value = 50;
            if (value < 50) value = 50;
            if (value > withdrawableAmount) value = withdrawableAmount;
            
            this.value = value.toFixed(2);
            slider.value = value;
            updateFeeCalculation(value);
        });
    }
    
    // Amount actions (25%, 50%, 75%, Max)
    amountActions.forEach(action => {
        action.addEventListener('click', function() {
            const percent = parseFloat(this.dataset.percent);
            let amount = 0;
            
            if (percent === 100) {
                amount = withdrawableAmount;
            } else {
                amount = (withdrawableAmount * percent) / 100;
            }
            
            // Round to nearest 50
            amount = Math.floor(amount / 50) * 50;
            if (amount < 50) amount = 50;
            
            amountInput.value = amount.toFixed(2);
            if (slider) slider.value = amount;
            updateFeeCalculation(amount);
            
            // Update active state
            amountActions.forEach(a => a.classList.remove('active'));
            this.classList.add('active');
        });
    });
    
    // Initial fee calculation
    updateFeeCalculation(50);
}

// Update fee calculation
function updateFeeCalculation(amount) {
    const networkFee = 1.00; // Fixed fee for TRC20
    const processingFee = amount * 0.01; // 1% processing fee
    const totalFees = networkFee + processingFee;
    const receiveAmount = amount - totalFees;
    
    document.getElementById('fee-amount').textContent = `$${parseFloat(amount).toFixed(2)}`;
    document.getElementById('fee-network').textContent = `$${networkFee.toFixed(2)}`;
    document.getElementById('fee-processing').textContent = `$${processingFee.toFixed(2)}`;
    document.getElementById('fee-total').textContent = `$${totalFees.toFixed(2)}`;
    document.getElementById('fee-receive').textContent = `$${receiveAmount.toFixed(2)}`;
}

// Withdrawal Form Submission
function initWithdrawalForm() {
    const form = document.getElementById('withdrawal-form');
    const submitBtn = form?.querySelector('.submit-btn');
    
    if (!form || !submitBtn) return;
    
    form.addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Get form values
        const walletAddress = document.getElementById('wallet-address').value.trim();
        const amount = parseFloat(document.getElementById('withdrawal-amount').value);
        const pin = document.getElementById('withdrawal-pin').value;
        const confirmCheckbox = document.getElementById('confirm-withdrawal');
        
        // Validation
        let isValid = true;
        let errorMessage = '';
        
        if (!walletAddress) {
            isValid = false;
            errorMessage = 'Wallet address is required';
            showFieldError('wallet-address', errorMessage);
        }
        
        if (!amount || amount < 50) {
            isValid = false;
            errorMessage = 'Minimum withdrawal amount is $50';
            showFieldError('withdrawal-amount', errorMessage);
        } else if (amount > 3250) {
            isValid = false;
            errorMessage = 'Maximum withdrawal amount is $3,250';
            showFieldError('withdrawal-amount', errorMessage);
        }
        
        if (!pin || pin.length !== 6) {
            isValid = false;
            errorMessage = 'Please enter a valid 6-digit PIN';
            showFieldError('withdrawal-pin', errorMessage);
        }
        
        if (!confirmCheckbox.checked) {
            isValid = false;
            if (window.CommonFunctions) {
                window.CommonFunctions.showErrorNotification('Please confirm the withdrawal terms');
            }
        }
        
        if (!isValid) return;
        
        // Clear errors
        clearAllErrors();
        
        // Show loading state
        if (window.CommonFunctions) {
            window.CommonFunctions.setButtonLoading(submitBtn, true);
        }
        
        // Simulate withdrawal request
        setTimeout(() => {
            // In a real application, this would be an API call
            console.log('Withdrawal requested:', {
                walletAddress,
                amount,
                pin
            });
            
            // Show success message
            if (window.CommonFunctions) {
                window.CommonFunctions.showSuccessNotification(`Withdrawal request submitted for $${amount}`);
            }
            
            // Reset form
            form.reset();
            updateFeeCalculation(50);
            document.querySelectorAll('.amount-action').forEach(a => a.classList.remove('active'));
            
            // Add to history table
            addToHistoryTable({
                walletAddress,
                amount,
                status: 'processing'
            });
            
            // Update dashboard stats
            updateDashboardStats(amount);
            
            // Reset loading state
            if (window.CommonFunctions) {
                window.CommonFunctions.setButtonLoading(submitBtn, false);
            }
        }, 2000);
    });
    
    // Forgot PIN button
    const forgotPinBtn = document.querySelector('.btn-forgot-pin');
    if (forgotPinBtn) {
        forgotPinBtn.addEventListener('click', function() {
            showForgotPinModal();
        });
    }
}

// Add to history table
function addToHistoryTable(withdrawal) {
    const tableBody = document.querySelector('.withdrawal-table tbody');
    if (!tableBody) return;
    
    const now = new Date();
    const dateStr = now.toLocaleDateString('en-US', { 
        month: 'short', 
        day: 'numeric', 
        year: 'numeric' 
    });
    const timeStr = now.toLocaleTimeString('en-US', { 
        hour: '2-digit', 
        minute: '2-digit' 
    });
    
    const shortAddress = withdrawal.walletAddress.length > 10 ? 
        withdrawal.walletAddress.substring(0, 10) + '...' : 
        withdrawal.walletAddress;
    
    const tr = document.createElement('tr');
    tr.innerHTML = `
        <td>
            <div class="date-cell">
                <span class="date">${dateStr}</span>
                <span class="time">${timeStr}</span>
            </div>
        </td>
        <td>
            <div class="amount-cell">
                <span class="crypto">${withdrawal.amount} USDT</span>
                <span class="fiat">$${withdrawal.amount.toFixed(2)}</span>
            </div>
        </td>
        <td>
            <div class="method-cell">
                <i class="fab fa-usd"></i>
                <span>USDT TRC20</span>
            </div>
        </td>
        <td>
            <div class="address-cell">
                <code class="address">${shortAddress}</code>
            </div>
        </td>
        <td>
            <div class="tx-cell">
                <code class="tx-hash">Pending</code>
            </div>
        </td>
        <td>
            <span class="status-badge processing">Processing</span>
        </td>
        <td>
            <button class="action-btn view-btn" title="View Details">
                <i class="fas fa-eye"></i>
            </button>
        </td>
    `;
    
    // Insert at the top
    tableBody.insertBefore(tr, tableBody.firstChild);
    
    // Initialize view button
    const viewBtn = tr.querySelector('.view-btn');
    if (viewBtn) {
        viewBtn.addEventListener('click', function() {
            showWithdrawalDetails({
                date: dateStr,
                time: timeStr,
                amount: withdrawal.amount,
                method: 'USDT TRC20',
                walletAddress: withdrawal.walletAddress,
                transactionHash: 'Pending',
                status: 'processing',
                networkFee: '1.00',
                processingFee: (withdrawal.amount * 0.01).toFixed(2),
                receiveAmount: (withdrawal.amount - 1 - (withdrawal.amount * 0.01)).toFixed(2)
            });
        });
    }
}

// Update dashboard stats
function updateDashboardStats(amount) {
    // Update available balance
    const balanceValue = document.querySelector('.stat-card:nth-child(1) .stat-value');
    if (balanceValue) {
        const currentBalance = parseFloat(balanceValue.textContent.replace('$', '').replace(',', ''));
        const newBalance = currentBalance - amount;
        balanceValue.textContent = `$${newBalance.toFixed(2)}`;
    }
    
    // Update withdrawable amount
    const withdrawableValue = document.querySelector('.stat-card:nth-child(3) .stat-value');
    if (withdrawableValue) {
        const currentWithdrawable = parseFloat(withdrawableValue.textContent.replace('$', '').replace(',', ''));
        const newWithdrawable = currentWithdrawable - amount;
        withdrawableValue.textContent = `$${newWithdrawable.toFixed(2)}`;
    }
    
    // Update pending withdrawals
    const pendingValue = document.querySelector('.stat-card:nth-child(4) .stat-value');
    if (pendingValue) {
        const currentPending = parseFloat(pendingValue.textContent.replace('$', '').replace(',', ''));
        const newPending = currentPending + amount;
        pendingValue.textContent = `$${newPending.toFixed(2)}`;
    }
}

// Table Actions Functionality
function initTableActions() {
    const viewButtons = document.querySelectorAll('.view-btn');
    
    viewButtons.forEach(button => {
        button.addEventListener('click', function() {
            const row = this.closest('tr');
            if (row) {
                const date = row.querySelector('.date').textContent;
                const time = row.querySelector('.time').textContent;
                const amount = row.querySelector('.crypto').textContent;
                const method = row.querySelector('.method-cell span').textContent;
                const address = row.querySelector('.address').textContent;
                const txHash = row.querySelector('.tx-hash').textContent;
                const status = row.querySelector('.status-badge').textContent.toLowerCase();
                
                // For demo purposes, use sample data
                const withdrawalData = {
                    date,
                    time,
                    amount: parseFloat(amount),
                    method,
                    walletAddress: address === '****1234' ? 'Bank Account ****1234' : 
                                 address.includes('...') ? 'TQrY4pFZK8r1gHt6WzXjM9oL3n2P5sR8dE' : address,
                    transactionHash: txHash === 'Pending' ? 'Processing...' : txHash,
                    status,
                    networkFee: '1.00',
                    processingFee: (parseFloat(amount) * 0.01).toFixed(2),
                    receiveAmount: (parseFloat(amount) - 1 - (parseFloat(amount) * 0.01)).toFixed(2)
                };
                
                showWithdrawalDetails(withdrawalData);
            }
        });
    });
}

// Show Withdrawal Details Modal
function showWithdrawalDetails(details) {
    const modal = document.getElementById('withdrawal-details-modal');
    const content = modal.querySelector('.withdrawal-details-content');
    
    if (!modal || !content) return;
    
    const statusClass = `status-${details.status}`;
    const statusText = details.status.charAt(0).toUpperCase() + details.status.slice(1);
    
    content.innerHTML = `
        <div class="detail-grid">
            <div class="detail-item">
                <span class="detail-label">Date & Time:</span>
                <span class="detail-value">${details.date} at ${details.time}</span>
            </div>
            <div class="detail-item">
                <span class="detail-label">Amount:</span>
                <span class="detail-value">$${details.amount.toFixed(2)}</span>
            </div>
            <div class="detail-item">
                <span class="detail-label">Method:</span>
                <span class="detail-value">${details.method}</span>
            </div>
            <div class="detail-item">
                <span class="detail-label">Status:</span>
                <span class="detail-value ${statusClass}">${statusText}</span>
            </div>
            <div class="detail-item full-width">
                <span class="detail-label">Wallet Address:</span>
                <div class="detail-value-with-copy">
                    <code>${details.walletAddress}</code>
                    <button class="copy-btn small" data-text="${details.walletAddress}">
                        <i class="fas fa-copy"></i>
                    </button>
                </div>
            </div>
            <div class="detail-item full-width">
                <span class="detail-label">Transaction Hash:</span>
                <div class="detail-value-with-copy">
                    <code>${details.transactionHash}</code>
                    ${details.transactionHash !== 'Processing...' ? 
                        `<button class="copy-btn small" data-text="${details.transactionHash}">
                            <i class="fas fa-copy"></i>
                        </button>` : ''}
                </div>
            </div>
            <div class="detail-item">
                <span class="detail-label">Network Fee:</span>
                <span class="detail-value">$${details.networkFee}</span>
            </div>
            <div class="detail-item">
                <span class="detail-label">Processing Fee:</span>
                <span class="detail-value">$${details.processingFee}</span>
            </div>
            <div class="detail-item total">
                <span class="detail-label">You Received:</span>
                <span class="detail-value highlight">$${details.receiveAmount}</span>
            </div>
        </div>
        
        <div class="modal-actions">
            <button type="button" class="btn-secondary close-modal">Close</button>
            ${details.status === 'completed' ? 
                `<button type="button" class="btn-primary" onclick="window.print()">
                    <i class="fas fa-print"></i> Print Receipt
                </button>` : ''}
        </div>
    `;
    
    // Show modal
    modal.style.display = 'flex';
    
    // Close functionality
    const closeBtn = modal.querySelector('.modal-close');
    const closeModalBtn = modal.querySelector('.close-modal');
    
    const closeModal = () => {
        modal.style.display = 'none';
    };
    
    if (closeBtn) closeBtn.addEventListener('click', closeModal);
    if (closeModalBtn) closeModalBtn.addEventListener('click', closeModal);
    
    // Close when clicking outside
    modal.addEventListener('click', function(e) {
        if (e.target === this) {
            closeModal();
        }
    });
    
    // Initialize copy buttons
    const copyButtons = content.querySelectorAll('.copy-btn.small');
    copyButtons.forEach(button => {
        button.addEventListener('click', function() {
            const text = this.dataset.text;
            if (text) {
                navigator.clipboard.writeText(text).then(() => {
                    const originalText = this.innerHTML;
                    this.innerHTML = '<i class="fas fa-check"></i>';
                    setTimeout(() => {
                        this.innerHTML = originalText;
                    }, 2000);
                });
            }
        });
    });
}

// Show Add Address Modal
function showAddAddressModal() {
    const modal = document.getElementById('add-address-modal');
    const form = document.getElementById('add-address-form');
    
    if (!modal || !form) return;
    
    // Reset form
    form.reset();
    
    // Show modal
    modal.style.display = 'flex';
    
    // Close functionality
    const closeBtn = modal.querySelector('.modal-close');
    const closeModalBtn = modal.querySelector('.close-modal');
    
    const closeModal = () => {
        modal.style.display = 'none';
    };
    
    if (closeBtn) closeBtn.addEventListener('click', closeModal);
    if (closeModalBtn) closeModalBtn.addEventListener('click', closeModal);
    
    // Form submission
    form.addEventListener('submit', function(e) {
        e.preventDefault();
        
        const alias = document.getElementById('address-alias').value.trim();
        const address = document.getElementById('new-address').value.trim();
        const method = document.getElementById('address-method').value;
        
        // Add to saved addresses
        addSavedAddress(alias, address, method);
        
        // Close modal
        closeModal();
        
        // Show success message
        if (window.CommonFunctions) {
            window.CommonFunctions.showSuccessNotification('Address saved successfully');
        }
    });
    
    // Close when clicking outside
    modal.addEventListener('click', function(e) {
        if (e.target === this) {
            closeModal();
        }
    });
}

// Add saved address
function addSavedAddress(alias, address, method) {
    const savedAddresses = document.querySelector('.saved-addresses');
    if (!savedAddresses) return;
    
    const methodIcons = {
        usdt: 'fab fa-usd',
        btc: 'fab fa-btc',
        eth: 'fab fa-ethereum'
    };
    
    const label = document.createElement('label');
    label.className = 'saved-address';
    label.innerHTML = `
        <input type="radio" name="saved-address" value="${address}">
        <div class="address-info">
            <span class="address-alias">${alias}</span>
            <span class="address-value">${address}</span>
        </div>
    `;
    
    // Insert before the "Add New Address" button
    const addButton = document.querySelector('.btn-add-address');
    if (addButton) {
        savedAddresses.insertBefore(label, addButton);
    } else {
        savedAddresses.appendChild(label);
    }
    
    // Initialize the radio button
    const radio = label.querySelector('input[type="radio"]');
    const walletAddressInput = document.getElementById('wallet-address');
    
    radio.addEventListener('change', function() {
        if (this.checked) {
            walletAddressInput.value = this.value;
        }
    });
}

// Show Forgot PIN Modal
function showForgotPinModal() {
    // Create modal
    const overlay = document.createElement('div');
    overlay.className = 'modal-overlay';
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.8);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: var(--z-modal);
        backdrop-filter: blur(5px);
    `;
    
    const modal = document.createElement('div');
    modal.className = 'modal-content';
    modal.innerHTML = `
        <div class="modal-header">
            <h3>Reset Security PIN</h3>
            <button class="modal-close">&times;</button>
        </div>
        <div class="modal-body">
            <p>For security reasons, PIN reset requires email verification.</p>
            
            <form id="reset-pin-form">
                <div class="form-group">
                    <label for="reset-email">Email Address</label>
                    <input type="email" id="reset-email" class="form-input" placeholder="you@example.com" required>
                </div>
                
                <div class="form-group">
                    <label class="checkbox-option">
                        <input type="checkbox" id="confirm-reset" required>
                        <span>I confirm I want to reset my security PIN</span>
                    </label>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="btn-secondary close-modal">Cancel</button>
                    <button type="submit" class="btn-primary">Send Reset Instructions</button>
                </div>
            </form>
        </div>
    `;
    
    // Style modal
    modal.style.cssText = `
        background: var(--dark-light);
        border-radius: var(--border-radius-xl);
        padding: var(--spacing-2xl);
        width: 90%;
        max-width: 500px;
        border: 1px solid rgba(255, 255, 255, 0.1);
        box-shadow: var(--shadow-xl);
    `;
    
    // Add to page
    overlay.appendChild(modal);
    document.body.appendChild(overlay);
    
    // Close functionality
    const closeBtn = modal.querySelector('.modal-close');
    const closeModalBtn = modal.querySelector('.close-modal');
    const form = modal.querySelector('#reset-pin-form');
    
    const closeModal = () => {
        document.body.removeChild(overlay);
    };
    
    if (closeBtn) closeBtn.addEventListener('click', closeModal);
    if (closeModalBtn) closeModalBtn.addEventListener('click', closeModal);
    
    overlay.addEventListener('click', function(e) {
        if (e.target === this) {
            closeModal();
        }
    });
    
    // Form submission
    if (form) {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            const email = document.getElementById('reset-email').value.trim();
            
            // Simulate sending reset instructions
            const submitBtn = form.querySelector('.btn-primary');
            const originalText = submitBtn.innerHTML;
            
            submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Sending...';
            submitBtn.disabled = true;
            
            setTimeout(() => {
                if (window.CommonFunctions) {
                    window.CommonFunctions.showSuccessNotification(`Reset instructions sent to ${email}`);
                }
                closeModal();
            }, 1500);
        });
    }
}

// Modals Initialization
function initModals() {
    const modals = ['withdrawal-details-modal', 'add-address-modal'];
    
    modals.forEach(modalId => {
        const modal = document.getElementById(modalId);
        if (modal) {
            // Close with Escape key
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape' && modal.style.display === 'flex') {
                    modal.style.display = 'none';
                }
            });
        }
    });
}

// Helper Functions
function showFieldError(fieldId, message) {
    const field = document.getElementById(fieldId);
    if (!field) return;
    
    // Add error styling
    field.style.borderColor = 'var(--danger-color)';
    
    // Show error message
    let errorElement = field.parentNode.querySelector('.error-message');
    if (!errorElement) {
        errorElement = document.createElement('div');
        errorElement.className = 'error-message';
        field.parentNode.appendChild(errorElement);
    }
    errorElement.textContent = message;
    errorElement.style.color = 'var(--danger-color)';
    errorElement.style.fontSize = 'var(--font-size-sm)';
    errorElement.style.marginTop = 'var(--spacing-xs)';
}

function clearFieldError(fieldId) {
    const field = document.getElementById(fieldId);
    if (!field) return;
    
    // Remove error styling
    field.style.borderColor = '';
    
    // Remove error message
    const errorElement = field.parentNode.querySelector('.error-message');
    if (errorElement) {
        errorElement.remove();
    }
}

function clearAllErrors() {
    const fields = ['wallet-address', 'withdrawal-amount', 'withdrawal-pin'];
    fields.forEach(clearFieldError);
}

// Filter functionality
const filterSelect = document.querySelector('.filter-select');
if (filterSelect) {
    filterSelect.addEventListener('change', function() {
        const status = this.value;
        const rows = document.querySelectorAll('.withdrawal-table tbody tr');
        
        rows.forEach(row => {
            const rowStatus = row.querySelector('.status-badge').textContent.toLowerCase();
            if (status === 'all' || rowStatus === status) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        });
    });
}