// Register Page JavaScript
let currentStep = 1;
const totalSteps = 3;

document.addEventListener('DOMContentLoaded', function() {
    // Initialize common functions
    if (window.CommonFunctions) {
        window.CommonFunctions.initPasswordToggle();
        window.CommonFunctions.initTooltips();
    }
    
    // Initialize registration form
    initRegistrationForm();
    
    // Initialize investment amount slider
    initAmountSlider();
    
    // Initialize password strength checker
    initPasswordStrength();
    
    // Update progress indicator
    updateProgressIndicator();
});

// Initialize registration form
function initRegistrationForm() {
    const registerForm = document.getElementById('registerForm');
    
    if (!registerForm) return;
    
    registerForm.addEventListener('submit', function(e) {
        e.preventDefault();
        
        // Validate all steps before submission
        if (!validateAllSteps()) {
            return;
        }
        
        // Get all form data
        const formData = getFormData();
        
        // Submit registration
        submitRegistration(formData);
    });
    
    // Add real-time validation
    setupRealTimeValidation();
}

// Initialize investment amount slider
function initAmountSlider() {
    const slider = document.getElementById('investment-amount');
    const amountValue = document.querySelector('.amount-value');
    
    if (!slider || !amountValue) return;
    
    // Format initial value
    if (window.CommonFunctions) {
        amountValue.textContent = window.CommonFunctions.formatCurrency(slider.value);
    }
    
    // Update display on slider change
    slider.addEventListener('input', function() {
        if (window.CommonFunctions) {
            amountValue.textContent = window.CommonFunctions.formatCurrency(this.value);
        }
    });
}

// Initialize password strength checker
function initPasswordStrength() {
    const passwordInput = document.getElementById('register-password');
    const strengthBar = document.getElementById('strength-bar');
    const strengthText = document.getElementById('strength-text');
    const confirmPasswordInput = document.getElementById('confirm-password');
    
    if (!passwordInput || !strengthBar || !strengthText) return;
    
    passwordInput.addEventListener('input', function() {
        const password = this.value;
        const strength = calculatePasswordStrength(password);
        
        // Update strength bar
        strengthBar.style.width = `${strength.score * 25}%`;
        
        // Update color based on strength
        if (strength.score === 0) {
            strengthBar.style.backgroundColor = 'transparent';
            strengthText.textContent = 'Password strength';
            strengthText.style.color = 'var(--gray-light)';
        } else if (strength.score === 1) {
            strengthBar.style.backgroundColor = 'var(--danger-color)';
            strengthText.textContent = 'Very Weak';
            strengthText.style.color = 'var(--danger-color)';
        } else if (strength.score === 2) {
            strengthBar.style.backgroundColor = 'var(--warning-color)';
            strengthText.textContent = 'Weak';
            strengthText.style.color = 'var(--warning-color)';
        } else if (strength.score === 3) {
            strengthBar.style.backgroundColor = 'var(--secondary-color)';
            strengthText.textContent = 'Good';
            strengthText.style.color = 'var(--secondary-color)';
        } else {
            strengthBar.style.backgroundColor = 'var(--success-color)';
            strengthText.textContent = 'Strong';
            strengthText.style.color = 'var(--success-color)';
        }
        
        // Update requirements checkmarks
        updatePasswordRequirements(password);
        
        // Check password match if confirm password is filled
        if (confirmPasswordInput && confirmPasswordInput.value) {
            checkPasswordMatch();
        }
    });
    
    // Check password match in real-time
    if (confirmPasswordInput) {
        confirmPasswordInput.addEventListener('input', checkPasswordMatch);
    }
}

// Setup real-time validation
function setupRealTimeValidation() {
    // Step 1 validation
    const firstName = document.getElementById('first-name');
    const lastName = document.getElementById('last-name');
    const email = document.getElementById('register-email');
    const phone = document.getElementById('phone');
    const country = document.getElementById('country');
    
    if (firstName) {
        firstName.addEventListener('blur', function() {
            if (this.value && this.value.length < 2) {
                showFieldError('first-name', 'First name must be at least 2 characters');
            } else {
                clearFieldError('first-name');
            }
        });
    }
    
    if (lastName) {
        lastName.addEventListener('blur', function() {
            if (this.value && this.value.length < 2) {
                showFieldError('last-name', 'Last name must be at least 2 characters');
            } else {
                clearFieldError('last-name');
            }
        });
    }
    
    if (email) {
        email.addEventListener('blur', function() {
            const emailValue = this.value.trim();
            if (emailValue && !isValidEmail(emailValue)) {
                showFieldError('register-email', 'Please enter a valid email address');
            } else {
                clearFieldError('register-email');
            }
        });
    }
    
    if (phone) {
        phone.addEventListener('blur', function() {
            const phoneValue = this.value;
            if (phoneValue && !isValidPhone(phoneValue)) {
                showFieldError('phone', 'Please enter a valid phone number');
            } else {
                clearFieldError('phone');
            }
        });
    }
    
    if (country) {
        country.addEventListener('change', function() {
            if (!this.value) {
                showFieldError('country', 'Please select your country');
            } else {
                clearFieldError('country');
            }
        });
    }
}

// Calculate password strength
function calculatePasswordStrength(password) {
    let score = 0;
    const feedback = [];
    
    // Length check
    if (password.length >= 8) score++;
    else feedback.push('At least 8 characters');
    
    // Lowercase check
    if (/[a-z]/.test(password)) score++;
    else feedback.push('Lowercase letters');
    
    // Uppercase check
    if (/[A-Z]/.test(password)) score++;
    else feedback.push('Uppercase letters');
    
    // Number check
    if (/\d/.test(password)) score++;
    else feedback.push('Numbers');
    
    // Special character check
    if (/[^A-Za-z0-9]/.test(password)) score++;
    else feedback.push('Special characters');
    
    return { score, feedback };
}

// Update password requirements checkmarks
function updatePasswordRequirements(password) {
    const requirements = document.querySelectorAll('.password-requirements p');
    
    if (requirements.length === 0) return;
    
    // Check each requirement
    const checks = [
        password.length >= 8,
        /[A-Z]/.test(password),
        /\d/.test(password),
        /[^A-Za-z0-9]/.test(password)
    ];
    
    requirements.forEach((req, index) => {
        const icon = req.querySelector('i');
        if (checks[index]) {
            icon.className = 'fas fa-check-circle';
            icon.style.color = 'var(--success-color)';
            req.classList.add('valid');
        } else {
            icon.className = 'fas fa-check-circle';
            icon.style.color = 'var(--gray-color)';
            req.classList.remove('valid');
        }
    });
}

// Check if passwords match
function checkPasswordMatch() {
    const password = document.getElementById('register-password').value;
    const confirmPassword = document.getElementById('confirm-password').value;
    const matchElement = document.getElementById('password-match');
    
    if (!matchElement) return;
    
    if (!confirmPassword) {
        matchElement.textContent = '';
        matchElement.className = 'validation-message';
        return;
    }
    
    if (password === confirmPassword) {
        matchElement.textContent = 'Passwords match';
        matchElement.className = 'validation-message success';
    } else {
        matchElement.textContent = 'Passwords do not match';
        matchElement.className = 'validation-message error';
    }
}

// Navigation between steps
function nextStep() {
    if (!validateStep(currentStep)) {
        return;
    }
    
    // Hide current step
    document.getElementById(`step${currentStep}`).style.display = 'none';
    
    // Show next step
    currentStep++;
    document.getElementById(`step${currentStep}`).style.display = 'block';
    
    // Update progress indicator
    updateProgressIndicator();
    
    // Scroll to top of form
    document.querySelector('.auth-form-section').scrollTop = 0;
}

function prevStep() {
    // Hide current step
    document.getElementById(`step${currentStep}`).style.display = 'none';
    
    // Show previous step
    currentStep--;
    document.getElementById(`step${currentStep}`).style.display = 'block';
    
    // Update progress indicator
    updateProgressIndicator();
    
    // Scroll to top of form
    document.querySelector('.auth-form-section').scrollTop = 0;
}

// Update progress indicator
function updateProgressIndicator() {
    const steps = document.querySelectorAll('.progress-step');
    
    steps.forEach((step, index) => {
        step.classList.remove('active');
        if (index + 1 <= currentStep) {
            step.classList.add('active');
        }
    });
}

// Validate current step
function validateStep(step) {
    let isValid = true;
    
    if (step === 1) {
        // Validate step 1 fields
        const firstName = document.getElementById('first-name').value.trim();
        const lastName = document.getElementById('last-name').value.trim();
        const email = document.getElementById('register-email').value.trim();
        const phone = document.getElementById('phone').value.trim();
        const country = document.getElementById('country').value;
        
        if (!firstName) {
            showFieldError('first-name', 'First name is required');
            isValid = false;
        } else if (firstName.length < 2) {
            showFieldError('first-name', 'First name must be at least 2 characters');
            isValid = false;
        }
        
        if (!lastName) {
            showFieldError('last-name', 'Last name is required');
            isValid = false;
        } else if (lastName.length < 2) {
            showFieldError('last-name', 'Last name must be at least 2 characters');
            isValid = false;
        }
        
        if (!email) {
            showFieldError('register-email', 'Email is required');
            isValid = false;
        } else if (!isValidEmail(email)) {
            showFieldError('register-email', 'Please enter a valid email address');
            isValid = false;
        }
        
        if (!phone) {
            showFieldError('phone', 'Phone number is required');
            isValid = false;
        } else if (!isValidPhone(phone)) {
            showFieldError('phone', 'Please enter a valid phone number');
            isValid = false;
        }
        
        if (!country) {
            showFieldError('country', 'Please select your country');
            isValid = false;
        }
    } else if (step === 2) {
        // Validate step 2 fields
        const password = document.getElementById('register-password').value;
        const confirmPassword = document.getElementById('confirm-password').value;
        const securityQuestion = document.getElementById('security-question').value;
        const securityAnswer = document.getElementById('security-answer').value.trim();
        
        if (!password) {
            showFieldError('register-password', 'Password is required');
            isValid = false;
        } else if (password.length < 8) {
            showFieldError('register-password', 'Password must be at least 8 characters');
            isValid = false;
        } else {
            const strength = calculatePasswordStrength(password);
            if (strength.score < 3) {
                showFieldError('register-password', 'Please choose a stronger password');
                isValid = false;
            }
        }
        
        if (!confirmPassword) {
            showFieldError('confirm-password', 'Please confirm your password');
            isValid = false;
        } else if (password !== confirmPassword) {
            showFieldError('confirm-password', 'Passwords do not match');
            isValid = false;
        }
        
        if (!securityQuestion) {
            showFieldError('security-question', 'Please select a security question');
            isValid = false;
        }
        
        if (!securityAnswer) {
            showFieldError('security-answer', 'Please provide an answer');
            isValid = false;
        } else if (securityAnswer.length < 2) {
            showFieldError('security-answer', 'Answer must be at least 2 characters');
            isValid = false;
        }
    }
    
    return isValid;
}

// Validate all steps before submission
function validateAllSteps() {
    // Validate each step
    for (let i = 1; i <= totalSteps; i++) {
        if (!validateStep(i)) {
            // Go to the step with error
            if (currentStep !== i) {
                // Hide all steps
                for (let j = 1; j <= totalSteps; j++) {
                    document.getElementById(`step${j}`).style.display = 'none';
                }
                
                // Show the step with error
                currentStep = i;
                document.getElementById(`step${i}`).style.display = 'block';
                updateProgressIndicator();
            }
            
            return false;
        }
    }
    
    // Additional validation for step 3
    const termsAgreed = document.getElementById('terms-agreement').checked;
    if (!termsAgreed) {
        showFieldError('terms-agreement', 'You must agree to the Terms of Service');
        return false;
    }
    
    return true;
}

// Get all form data
function getFormData() {
    return {
        // Step 1 data
        firstName: document.getElementById('first-name').value.trim(),
        lastName: document.getElementById('last-name').value.trim(),
        email: document.getElementById('register-email').value.trim(),
        phone: document.getElementById('phone').value.trim(),
        country: document.getElementById('country').value,
        
        // Step 2 data
        password: document.getElementById('register-password').value,
        securityQuestion: document.getElementById('security-question').value,
        securityAnswer: document.getElementById('security-answer').value.trim(),
        
        // Step 3 data
        experience: document.querySelector('input[name="experience"]:checked')?.value,
        goal: document.querySelector('input[name="goal"]:checked')?.value,
        investmentAmount: document.getElementById('investment-amount').value,
        newsletter: document.getElementById('newsletter').checked,
        marketing: document.getElementById('marketing').checked,
        termsAgreed: document.getElementById('terms-agreement').checked
    };
}

// Submit registration
function submitRegistration(formData) {
    const submitBtn = document.querySelector('#registerForm .submit-btn[type="submit"]');
    
    // Show loading state
    if (window.CommonFunctions) {
        window.CommonFunctions.setButtonLoading(submitBtn, true);
    }
    
    // Simulate API call delay
    setTimeout(() => {
        // In a real application, this would be an API call to your backend
        console.log('Registration data:', formData);
        
        // For demo purposes, simulate successful registration
        const isSuccess = Math.random() > 0.1; // 90% success rate for demo
        
        if (isSuccess) {
            // Show success message
            if (window.CommonFunctions) {
                const message = `Welcome ${formData.firstName}! Your investment account has been created.`;
                window.CommonFunctions.showSuccessNotification(message);
            }
            
            // Show account details modal
            showAccountDetailsModal(formData);
        } else {
            // Show error message
            if (window.CommonFunctions) {
                window.CommonFunctions.showErrorNotification('Registration failed. Please try again.');
            }
        }
        
        // Reset loading state
        if (window.CommonFunctions) {
            window.CommonFunctions.setButtonLoading(submitBtn, false);
        }
    }, 2000);
}

// Show account details modal
function showAccountDetailsModal(formData) {
    // Create modal overlay
    const overlay = document.createElement('div');
    overlay.className = 'modal-overlay';
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.9);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: var(--z-modal);
        backdrop-filter: blur(5px);
    `;
    
    // Create modal content
    const modal = document.createElement('div');
    modal.className = 'modal-content';
    modal.innerHTML = `
        <div class="modal-header">
            <h3>🎉 Welcome to Freenexa!</h3>
            <button class="modal-close">&times;</button>
        </div>
        <div class="modal-body">
            <div class="success-icon">
                <i class="fas fa-check-circle"></i>
            </div>
            <h4>Your investment account is ready!</h4>
            <p>Here are your account details:</p>
            
            <div class="account-details">
                <div class="detail-item">
                    <span class="detail-label">Account Number:</span>
                    <span class="detail-value">DV-${Math.floor(Math.random() * 1000000).toString().padStart(6, '0')}</span>
                </div>
                <div class="detail-item">
                    <span class="detail-label">Name:</span>
                    <span class="detail-value">${formData.firstName} ${formData.lastName}</span>
                </div>
                <div class="detail-item">
                    <span class="detail-label">Email:</span>
                    <span class="detail-value">${formData.email}</span>
                </div>
                <div class="detail-item">
                    <span class="detail-label">Initial Investment:</span>
                    <span class="detail-value">${window.CommonFunctions ? window.CommonFunctions.formatCurrency(formData.investmentAmount) : '$' + formData.investmentAmount}</span>
                </div>
            </div>
            
            <div class="next-steps">
                <h5>Next Steps:</h5>
                <ol>
                    <li>Verify your email address (check your inbox)</li>
                    <li>Complete your investor profile</li>
                    <li>Fund your account to start investing</li>
                    <li>Explore our investment portfolios</li>
                </ol>
            </div>
            
            <div class="modal-actions">
                <button class="btn-secondary" id="go-to-login">Go to Login</button>
                <button class="btn-primary" id="explore-dashboard">Explore Dashboard</button>
            </div>
        </div>
    `;
    
    // Style modal
    modal.style.cssText = `
        background: var(--dark-light);
        border-radius: var(--border-radius-xl);
        padding: var(--spacing-2xl);
        width: 90%;
        max-width: 600px;
        border: 1px solid rgba(255, 255, 255, 0.1);
        box-shadow: var(--shadow-xl);
        text-align: center;
    `;
    
    // Style modal header
    const modalHeader = modal.querySelector('.modal-header');
    modalHeader.style.cssText = `
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: var(--spacing-xl);
        padding-bottom: var(--spacing-md);
        border-bottom: 1px solid rgba(255, 255, 255, 0.1);
    `;
    
    // Style success icon
    const successIcon = modal.querySelector('.success-icon');
    successIcon.style.cssText = `
        font-size: 4rem;
        color: var(--success-color);
        margin: var(--spacing-xl) 0;
    `;
    
    // Style account details
    const accountDetails = modal.querySelector('.account-details');
    accountDetails.style.cssText = `
        background: rgba(255, 255, 255, 0.05);
        border-radius: var(--border-radius-lg);
        padding: var(--spacing-lg);
        margin: var(--spacing-xl) 0;
        text-align: left;
    `;
    
    const detailItems = modal.querySelectorAll('.detail-item');
    detailItems.forEach(item => {
        item.style.cssText = `
            display: flex;
            justify-content: space-between;
            padding: var(--spacing-sm) 0;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        `;
        
        const detailLabel = item.querySelector('.detail-label');
        detailLabel.style.cssText = `
            color: var(--gray-light);
            font-weight: 500;
        `;
        
        const detailValue = item.querySelector('.detail-value');
        detailValue.style.cssText = `
            color: var(--light-color);
            font-weight: 600;
        `;
    });
    
    // Style next steps
    const nextSteps = modal.querySelector('.next-steps');
    nextSteps.style.cssText = `
        text-align: left;
        margin: var(--spacing-xl) 0;
    `;
    
    nextSteps.querySelector('h5').style.cssText = `
        color: var(--light-color);
        margin-bottom: var(--spacing-md);
    `;
    
    nextSteps.querySelector('ol').style.cssText = `
        color: var(--gray-light);
        padding-left: var(--spacing-lg);
        line-height: 1.8;
    `;
    
    // Style modal actions
    const modalActions = modal.querySelector('.modal-actions');
    modalActions.style.cssText = `
        display: flex;
        gap: var(--spacing-md);
        margin-top: var(--spacing-xl);
    `;
    
    const buttons = modalActions.querySelectorAll('button');
    buttons.forEach(btn => {
        btn.style.cssText = `
            flex: 1;
            padding: var(--spacing-md) var(--spacing-lg);
            border-radius: var(--border-radius-md);
            font-family: var(--font-primary);
            font-size: var(--font-size-base);
            font-weight: 600;
            cursor: pointer;
            transition: all var(--transition-normal);
        `;
    });
    
    const btnSecondary = modal.querySelector('#go-to-login');
    btnSecondary.style.cssText += `
        background: rgba(255, 255, 255, 0.1);
        color: var(--light-color);
        border: 1px solid rgba(255, 255, 255, 0.2);
    `;
    
    const btnPrimary = modal.querySelector('#explore-dashboard');
    btnPrimary.style.cssText += `
        background: linear-gradient(to right, var(--primary-color), var(--primary-dark));
        color: white;
        border: none;
    `;
    
    // Style modal close button
    const closeBtn = modal.querySelector('.modal-close');
    closeBtn.style.cssText = `
        background: none;
        border: none;
        color: var(--light-color);
        font-size: var(--font-size-2xl);
        cursor: pointer;
        padding: 0;
        width: 30px;
        height: 30px;
        display: flex;
        align-items: center;
        justify-content: center;
        border-radius: var(--border-radius-circle);
        transition: background-color var(--transition-fast);
    `;
    
    // Add event listeners
    closeBtn.addEventListener('mouseenter', function() {
        this.style.background = 'rgba(255, 255, 255, 0.1)';
    });
    
    closeBtn.addEventListener('mouseleave', function() {
        this.style.background = 'none';
    });
    
    closeBtn.addEventListener('click', function() {
        document.body.removeChild(overlay);
        // Redirect to login page
        setTimeout(() => {
            window.location.href = 'index.html';
        }, 300);
    });
    
    overlay.addEventListener('click', function(e) {
        if (e.target === overlay) {
            document.body.removeChild(overlay);
            // Redirect to login page
            setTimeout(() => {
                window.location.href = 'index.html';
            }, 300);
        }
    });
    
    btnSecondary.addEventListener('click', function() {
        document.body.removeChild(overlay);
        window.location.href = 'index.html';
    });
    
    btnPrimary.addEventListener('click', function() {
        // In a real app, redirect to dashboard
        alert('In a real application, you would be redirected to your investment dashboard.');
        document.body.removeChild(overlay);
        window.location.href = 'index.html';
    });
    
    // Add modal to page
    overlay.appendChild(modal);
    document.body.appendChild(overlay);
}

// Helper functions
function showFieldError(fieldId, message) {
    const field = document.getElementById(fieldId);
    if (!field) return;
    
    // Add error class to input
    field.classList.add('error');
    
    // Show error message
    if (window.CommonFunctions) {
        window.CommonFunctions.showValidationMessage(field, message, 'error');
    }
}

function clearFieldError(fieldId) {
    const field = document.getElementById(fieldId);
    if (!field) return;
    
    // Remove error class
    field.classList.remove('error');
    
    // Clear error message
    const errorMessage = field.parentNode.querySelector('.validation-message.error');
    if (errorMessage) {
        errorMessage.remove();
    }
}

function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

function isValidPhone(phone) {
    // Remove all non-digit characters
    const digits = phone.replace(/\D/g, '');
    return digits.length >= 10;
}