// Deposit Page JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // Initialize deposit functionality
    initDepositPage();
});

function initDepositPage() {
    // Initialize method tabs
    initMethodTabs();
    
    // Initialize copy buttons
    initCopyButtons();
    
    // Initialize amount slider
    initAmountSlider();
    
    // Initialize file upload
    initFileUpload();
    
    // Initialize deposit form submission
    initDepositForm();
    
    // Initialize table actions
    initTableActions();
    
    // Initialize modal
    initModal();
}

// Method Tabs Functionality
function initMethodTabs() {
    const tabs = document.querySelectorAll('.method-tab');
    const contents = document.querySelectorAll('.method-content');
    
    tabs.forEach(tab => {
        tab.addEventListener('click', function() {
            const method = this.dataset.method;
            
            // Remove active class from all tabs and contents
            tabs.forEach(t => t.classList.remove('active'));
            contents.forEach(c => c.classList.remove('active'));
            
            // Add active class to current tab
            this.classList.add('active');
            
            // Show corresponding content
            const content = document.getElementById(`${method}-method`);
            if (content) {
                content.classList.add('active');
            }
        });
    });
}

// Copy Buttons Functionality
function initCopyButtons() {
    const copyButtons = document.querySelectorAll('.copy-btn, .copy-small');
    
    copyButtons.forEach(button => {
        button.addEventListener('click', function() {
            let textToCopy = '';
            
            if (this.classList.contains('copy-btn')) {
                const target = this.dataset.clipboardTarget;
                const input = document.querySelector(target);
                if (input) {
                    textToCopy = input.value;
                }
            } else if (this.classList.contains('copy-small')) {
                const txHash = this.parentNode.querySelector('.tx-hash');
                if (txHash) {
                    textToCopy = txHash.textContent;
                }
            }
            
            if (textToCopy) {
                navigator.clipboard.writeText(textToCopy).then(() => {
                    // Show success feedback
                    const originalText = this.innerHTML;
                    this.innerHTML = '<i class="fas fa-check"></i> Copied!';
                    this.classList.add('copied');
                    
                    // Reset after 2 seconds
                    setTimeout(() => {
                        this.innerHTML = originalText;
                        this.classList.remove('copied');
                    }, 2000);
                    
                    // Show notification
                    if (window.CommonFunctions) {
                        window.CommonFunctions.showSuccessNotification('Copied to clipboard!');
                    }
                }).catch(err => {
                    console.error('Failed to copy: ', err);
                    if (window.CommonFunctions) {
                        window.CommonFunctions.showErrorNotification('Failed to copy text');
                    }
                });
            }
        });
    });
}

// Amount Slider Functionality
function initAmountSlider() {
    const slider = document.getElementById('amount-slider');
    const amountInput = document.getElementById('deposit-amount');
    
    if (slider && amountInput) {
        // Update input when slider changes
        slider.addEventListener('input', function() {
            amountInput.value = this.value;
        });
        
        // Update slider when input changes
        amountInput.addEventListener('input', function() {
            let value = parseFloat(this.value);
            if (isNaN(value)) value = 100;
            if (value < 100) value = 100;
            if (value > 10000) value = 10000;
            
            slider.value = value;
            this.value = value.toFixed(2);
        });
        
        // Format input on blur
        amountInput.addEventListener('blur', function() {
            let value = parseFloat(this.value);
            if (isNaN(value)) value = 100;
            if (value < 100) value = 100;
            if (value > 10000) value = 10000;
            
            this.value = value.toFixed(2);
            slider.value = value;
        });
    }
}

// File Upload Functionality
function initFileUpload() {
    const fileInput = document.getElementById('payment-proof');
    const uploadArea = document.querySelector('.upload-area');
    
    if (fileInput && uploadArea) {
        // Click upload area to trigger file input
        uploadArea.addEventListener('click', function() {
            fileInput.click();
        });
        
        // Handle file selection
        fileInput.addEventListener('change', function() {
            if (this.files && this.files[0]) {
                const file = this.files[0];
                const fileSize = file.size / 1024 / 1024; // Convert to MB
                
                // Check file size (5MB limit)
                if (fileSize > 5) {
                    if (window.CommonFunctions) {
                        window.CommonFunctions.showErrorNotification('File size must be less than 5MB');
                    }
                    this.value = '';
                    return;
                }
                
                // Check file type
                const validTypes = ['image/jpeg', 'image/png', 'image/jpg', 'application/pdf'];
                if (!validTypes.includes(file.type)) {
                    if (window.CommonFunctions) {
                        window.CommonFunctions.showErrorNotification('Only JPG, PNG, and PDF files are allowed');
                    }
                    this.value = '';
                    return;
                }
                
                // Update upload area to show selected file
                const fileName = file.name.length > 20 ? 
                    file.name.substring(0, 20) + '...' : file.name;
                
                uploadArea.innerHTML = `
                    <i class="fas fa-file-alt"></i>
                    <p>${fileName}</p>
                    <p class="file-types">${(fileSize).toFixed(2)} MB</p>
                `;
                
                uploadArea.style.borderColor = 'var(--success-color)';
                uploadArea.style.background = 'rgba(40, 167, 69, 0.1)';
                
                if (window.CommonFunctions) {
                    window.CommonFunctions.showSuccessNotification('File selected successfully');
                }
            }
        });
        
        // Drag and drop functionality
        uploadArea.addEventListener('dragover', function(e) {
            e.preventDefault();
            this.style.borderColor = 'var(--primary-light)';
            this.style.background = 'rgba(255, 255, 255, 0.08)';
        });
        
        uploadArea.addEventListener('dragleave', function(e) {
            e.preventDefault();
            this.style.borderColor = 'rgba(255, 255, 255, 0.2)';
            this.style.background = 'rgba(255, 255, 255, 0.05)';
        });
        
        uploadArea.addEventListener('drop', function(e) {
            e.preventDefault();
            this.style.borderColor = 'rgba(255, 255, 255, 0.2)';
            this.style.background = 'rgba(255, 255, 255, 0.05)';
            
            if (e.dataTransfer.files.length) {
                fileInput.files = e.dataTransfer.files;
                const event = new Event('change', { bubbles: true });
                fileInput.dispatchEvent(event);
            }
        });
    }
}

// Deposit Form Submission
function initDepositForm() {
    const form = document.querySelector('.deposit-form');
    const submitBtn = document.getElementById('submit-deposit');
    
    if (form && submitBtn) {
        submitBtn.addEventListener('click', function(e) {
            e.preventDefault();
            
            // Get form values
            const amount = parseFloat(document.getElementById('deposit-amount').value);
            const transactionHash = document.getElementById('transaction-hash').value.trim();
            const walletAddress = document.getElementById('wallet-address').value.trim();
            const confirmTerms = document.getElementById('confirm-terms').checked;
            
            // Validation
            let isValid = true;
            let errorMessage = '';
            
            if (!amount || amount < 100) {
                isValid = false;
                errorMessage = 'Minimum deposit amount is $100 USDT';
                showFieldError('deposit-amount', errorMessage);
            }
            
            if (!transactionHash) {
                isValid = false;
                errorMessage = 'Transaction ID/Hash is required';
                showFieldError('transaction-hash', errorMessage);
            }
            
            if (!walletAddress) {
                isValid = false;
                errorMessage = 'Wallet address is required';
                showFieldError('wallet-address', errorMessage);
            }
            
            if (!confirmTerms) {
                isValid = false;
                if (window.CommonFunctions) {
                    window.CommonFunctions.showErrorNotification('Please confirm the deposit terms');
                }
            }
            
            if (!isValid) return;
            
            // Clear all errors
            clearAllErrors();
            
            // Show loading state
            if (window.CommonFunctions) {
                window.CommonFunctions.setButtonLoading(submitBtn, true);
            }
            
            // Simulate deposit submission
            setTimeout(() => {
                // In a real application, this would be an API call
                console.log('Deposit submitted:', {
                    amount,
                    transactionHash,
                    walletAddress
                });
                
                // Show success message
                if (window.CommonFunctions) {
                    window.CommonFunctions.showSuccessNotification(`Deposit request submitted for $${amount} USDT`);
                }
                
                // Reset form
                form.reset();
                document.querySelector('.upload-area').innerHTML = `
                    <i class="fas fa-cloud-upload-alt"></i>
                    <p>Drag & drop or <span class="browse-link">browse</span></p>
                    <p class="file-types">JPG, PNG, PDF up to 5MB</p>
                `;
                
                // Reset loading state
                if (window.CommonFunctions) {
                    window.CommonFunctions.setButtonLoading(submitBtn, false);
                }
                
                // Add to history table (demo only)
                addToHistoryTable({
                    amount,
                    transactionHash,
                    walletAddress
                });
            }, 1500);
        });
    }
}

// Add to history table (demo)
function addToHistoryTable(deposit) {
    const tableBody = document.querySelector('.deposit-table tbody');
    if (!tableBody) return;
    
    const now = new Date();
    const dateStr = now.toLocaleDateString('en-US', { 
        month: 'short', 
        day: 'numeric', 
        year: 'numeric' 
    });
    const timeStr = now.toLocaleTimeString('en-US', { 
        hour: '2-digit', 
        minute: '2-digit' 
    });
    
    const tr = document.createElement('tr');
    tr.innerHTML = `
        <td>
            <div class="date-cell">
                <span class="date">${dateStr}</span>
                <span class="time">${timeStr}</span>
            </div>
        </td>
        <td>
            <div class="amount-cell">
                <span class="crypto">${deposit.amount} USDT</span>
                <span class="fiat">$${deposit.amount.toFixed(2)}</span>
            </div>
        </td>
        <td>
            <div class="method-cell">
                <i class="fab fa-usd"></i>
                <span>USDT TRC20</span>
            </div>
        </td>
        <td>
            <div class="tx-cell">
                <code class="tx-hash">${deposit.transactionHash.substring(0, 10)}...</code>
                <button class="copy-small" title="Copy">
                    <i class="fas fa-copy"></i>
                </button>
            </div>
        </td>
        <td>
            <span class="status-badge pending">Pending</span>
        </td>
        <td>
            <button class="action-btn view-btn" title="View Details">
                <i class="fas fa-eye"></i>
            </button>
        </td>
    `;
    
    // Insert at the top
    tableBody.insertBefore(tr, tableBody.firstChild);
    
    // Reinitialize copy button for the new row
    const newCopyBtn = tr.querySelector('.copy-small');
    if (newCopyBtn) {
        newCopyBtn.addEventListener('click', function() {
            navigator.clipboard.writeText(deposit.transactionHash).then(() => {
                const originalText = this.innerHTML;
                this.innerHTML = '<i class="fas fa-check"></i>';
                setTimeout(() => {
                    this.innerHTML = originalText;
                }, 2000);
            });
        });
    }
    
    // Reinitialize view button for the new row
    const newViewBtn = tr.querySelector('.view-btn');
    if (newViewBtn) {
        newViewBtn.addEventListener('click', function() {
            showDepositDetails({
                date: dateStr,
                time: timeStr,
                amount: deposit.amount,
                method: 'USDT TRC20',
                transactionHash: deposit.transactionHash,
                walletAddress: deposit.walletAddress,
                status: 'pending'
            });
        });
    }
}

// Table Actions Functionality
function initTableActions() {
    // View buttons
    const viewButtons = document.querySelectorAll('.view-btn');
    viewButtons.forEach(button => {
        button.addEventListener('click', function() {
            const row = this.closest('tr');
            if (row) {
                const date = row.querySelector('.date').textContent;
                const time = row.querySelector('.time').textContent;
                const amount = row.querySelector('.crypto').textContent;
                const method = row.querySelector('.method-cell span').textContent;
                const txHash = row.querySelector('.tx-hash').textContent;
                const status = row.querySelector('.status-badge').textContent;
                
                // Get wallet address from a data attribute (would be populated from server)
                const walletAddress = 'TQrY4pFZK8r1gHt6WzXjM9oL3n2P5sR8dE';
                
                showDepositDetails({
                    date,
                    time,
                    amount,
                    method,
                    transactionHash: txHash,
                    walletAddress,
                    status
                });
            }
        });
    });
}

// Show Deposit Details Modal
function showDepositDetails(details) {
    const modal = document.getElementById('deposit-details-modal');
    const content = modal.querySelector('.deposit-details-content');
    
    if (!modal || !content) return;
    
    // Populate content
    content.innerHTML = `
        <div class="detail-grid">
            <div class="detail-item">
                <span class="detail-label">Date & Time:</span>
                <span class="detail-value">${details.date} at ${details.time}</span>
            </div>
            <div class="detail-item">
                <span class="detail-label">Amount:</span>
                <span class="detail-value">${details.amount}</span>
            </div>
            <div class="detail-item">
                <span class="detail-label">Method:</span>
                <span class="detail-value">${details.method}</span>
            </div>
            <div class="detail-item">
                <span class="detail-label">Status:</span>
                <span class="detail-value status-${details.status}">${details.status.charAt(0).toUpperCase() + details.status.slice(1)}</span>
            </div>
            <div class="detail-item full-width">
                <span class="detail-label">Transaction Hash:</span>
                <div class="detail-value-with-copy">
                    <code>${details.transactionHash}</code>
                    <button class="copy-btn small" data-text="${details.transactionHash}">
                        <i class="fas fa-copy"></i>
                    </button>
                </div>
            </div>
            <div class="detail-item full-width">
                <span class="detail-label">Wallet Address:</span>
                <div class="detail-value-with-copy">
                    <code>${details.walletAddress}</code>
                    <button class="copy-btn small" data-text="${details.walletAddress}">
                        <i class="fas fa-copy"></i>
                    </button>
                </div>
            </div>
        </div>
        
        <div class="modal-actions">
            <button class="btn-secondary close-modal">Close</button>
            <button class="btn-primary" onclick="window.print()">
                <i class="fas fa-print"></i> Print Receipt
            </button>
        </div>
    `;
    
    // Show modal
    modal.style.display = 'flex';
    
    // Add close functionality
    const closeBtn = modal.querySelector('.modal-close');
    const closeModalBtn = modal.querySelector('.close-modal');
    
    const closeModal = () => {
        modal.style.display = 'none';
    };
    
    if (closeBtn) closeBtn.addEventListener('click', closeModal);
    if (closeModalBtn) closeModalBtn.addEventListener('click', closeModal);
    
    // Close when clicking outside
    modal.addEventListener('click', function(e) {
        if (e.target === this) {
            closeModal();
        }
    });
    
    // Initialize copy buttons in modal
    const copyButtons = content.querySelectorAll('.copy-btn.small');
    copyButtons.forEach(button => {
        button.addEventListener('click', function() {
            const text = this.dataset.text;
            if (text) {
                navigator.clipboard.writeText(text).then(() => {
                    const originalText = this.innerHTML;
                    this.innerHTML = '<i class="fas fa-check"></i>';
                    setTimeout(() => {
                        this.innerHTML = originalText;
                    }, 2000);
                });
            }
        });
    });
}

// Modal Initialization
function initModal() {
    const modal = document.getElementById('deposit-details-modal');
    if (modal) {
        // Close modal with Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && modal.style.display === 'flex') {
                modal.style.display = 'none';
            }
        });
    }
}

// Helper Functions
function showFieldError(fieldId, message) {
    const field = document.getElementById(fieldId);
    if (!field) return;
    
    // Add error styling
    field.style.borderColor = 'var(--danger-color)';
    
    // Show error message
    let errorElement = field.parentNode.querySelector('.error-message');
    if (!errorElement) {
        errorElement = document.createElement('div');
        errorElement.className = 'error-message';
        field.parentNode.appendChild(errorElement);
    }
    errorElement.textContent = message;
    errorElement.style.color = 'var(--danger-color)';
    errorElement.style.fontSize = 'var(--font-size-sm)';
    errorElement.style.marginTop = 'var(--spacing-xs)';
}

function clearFieldError(fieldId) {
    const field = document.getElementById(fieldId);
    if (!field) return;
    
    // Remove error styling
    field.style.borderColor = '';
    
    // Remove error message
    const errorElement = field.parentNode.querySelector('.error-message');
    if (errorElement) {
        errorElement.remove();
    }
}

function clearAllErrors() {
    const fields = ['deposit-amount', 'transaction-hash', 'wallet-address'];
    fields.forEach(clearFieldError);
}

// Filter functionality
const filterSelect = document.querySelector('.filter-select');
if (filterSelect) {
    filterSelect.addEventListener('change', function() {
        const status = this.value;
        const rows = document.querySelectorAll('.deposit-table tbody tr');
        
        rows.forEach(row => {
            const rowStatus = row.querySelector('.status-badge').textContent.toLowerCase();
            if (status === 'all' || rowStatus === status) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        });
    });
}