// Dashboard JavaScript
document.addEventListener('DOMContentLoaded', function() {
    // Initialize dashboard functionality
    initDashboard();
});

function initDashboard() {
    // Initialize mobile menu
    initMobileMenu();
    
    // Initialize portfolio chart
    initPortfolioChart();
    
    // Initialize calculator
    initDashboardCalculator();
    
    // Initialize time filter
    initTimeFilter();
    
    // Initialize security status
    initSecurityStatus();
    
    // Initialize market data
    initMarketData();
    
    // Initialize transactions
    initTransactions();
    
    // Initialize quick actions
    initQuickActions();
    
    // Initialize animations
    initAnimations();
}

// Mobile Menu Functionality
function initMobileMenu() {
    const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
    const closeSidebar = document.querySelector('.close-sidebar');
    const mobileSidebar = document.querySelector('.mobile-sidebar');
    const sidebarLinks = document.querySelectorAll('.sidebar-link');
    
    if (mobileMenuBtn && mobileSidebar) {
        mobileMenuBtn.addEventListener('click', function() {
            mobileSidebar.classList.add('active');
            document.body.style.overflow = 'hidden';
        });
        
        closeSidebar.addEventListener('click', function() {
            mobileSidebar.classList.remove('active');
            document.body.style.overflow = '';
        });
        
        // Close sidebar when clicking on links
        sidebarLinks.forEach(link => {
            link.addEventListener('click', function() {
                mobileSidebar.classList.remove('active');
                document.body.style.overflow = '';
            });
        });
        
        // Close sidebar when clicking outside
        document.addEventListener('click', function(event) {
            if (!mobileSidebar.contains(event.target) && 
                !mobileMenuBtn.contains(event.target) && 
                mobileSidebar.classList.contains('active')) {
                mobileSidebar.classList.remove('active');
                document.body.style.overflow = '';
            }
        });
        
        // Close with Escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && mobileSidebar.classList.contains('active')) {
                mobileSidebar.classList.remove('active');
                document.body.style.overflow = '';
            }
        });
    }
}

// Portfolio Chart
let portfolioChart = null;

function initPortfolioChart() {
    const ctx = document.getElementById('portfolio-chart').getContext('2d');
    
    // Sample data for last 30 days
    const labels = [];
    const data = [];
    
    // Generate last 30 days labels
    for (let i = 30; i >= 0; i--) {
        const date = new Date();
        date.setDate(date.getDate() - i);
        labels.push(date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' }));
    }
    
    // Generate sample portfolio data (starting from $10,000 with daily variations)
    let currentValue = 10000;
    for (let i = 0; i <= 30; i++) {
        // Random daily change between -2% and +3%
        const dailyChange = (Math.random() * 5 - 2) / 100;
        currentValue *= (1 + dailyChange);
        data.push(currentValue);
    }
    
    portfolioChart = new Chart(ctx, {
        type: 'line',
        data: {
            labels: labels,
            datasets: [{
                label: 'Portfolio Value',
                data: data,
                borderColor: 'rgba(87, 193, 232, 1)',
                backgroundColor: 'rgba(87, 193, 232, 0.1)',
                borderWidth: 2,
                fill: true,
                tension: 0.4,
                pointRadius: 0,
                pointHoverRadius: 6
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                },
                tooltip: {
                    mode: 'index',
                    intersect: false,
                    callbacks: {
                        label: function(context) {
                            return `Value: $${context.parsed.y.toFixed(2)}`;
                        }
                    }
                }
            },
            scales: {
                y: {
                    beginAtZero: false,
                    grid: {
                        color: 'rgba(255, 255, 255, 0.1)'
                    },
                    ticks: {
                        color: 'rgba(255, 255, 255, 0.7)',
                        callback: function(value) {
                            return '$' + value.toLocaleString();
                        }
                    }
                },
                x: {
                    grid: {
                        color: 'rgba(255, 255, 255, 0.1)'
                    },
                    ticks: {
                        color: 'rgba(255, 255, 255, 0.7)',
                        maxTicksLimit: 8
                    }
                }
            },
            interaction: {
                intersect: false,
                mode: 'nearest'
            },
            elements: {
                line: {
                    cubicInterpolationMode: 'monotone'
                }
            }
        }
    });
}

// Dashboard Calculator
function initDashboardCalculator() {
    const amountInput = document.querySelector('.calculator .amount-input input');
    const planSelect = document.querySelector('.calculator select');
    const dailyProfit = document.querySelector('.calculator .result:nth-child(1) .value');
    const monthlyProfit = document.querySelector('.calculator .result:nth-child(2) .value');
    const totalReturn = document.querySelector('.calculator .result:nth-child(3) .value');
    
    // Plan configurations
    const plans = {
        '30': { dailyReturn: 0.01, duration: 30 },
        '180': { dailyReturn: 0.012, duration: 180 },
        '365': { dailyReturn: 0.015, duration: 365 }
    };
    
    function updateCalculator() {
        const amount = parseFloat(amountInput.value) || 1000;
        const plan = planSelect.value;
        const planData = plans[plan];
        
        const daily = amount * planData.dailyReturn;
        const monthly = daily * 30;
        const total = amount + (daily * planData.duration);
        
        dailyProfit.textContent = `$${daily.toFixed(2)}`;
        monthlyProfit.textContent = `$${monthly.toFixed(2)}`;
        totalReturn.textContent = `$${total.toFixed(2)}`;
    }
    
    if (amountInput && planSelect) {
        amountInput.addEventListener('input', updateCalculator);
        planSelect.addEventListener('change', updateCalculator);
        
        // Initial calculation
        updateCalculator();
    }
}

// Time Filter Functionality
function initTimeFilter() {
    const timeFilter = document.querySelector('.time-filter');
    
    if (timeFilter) {
        timeFilter.addEventListener('change', function() {
            const period = this.value;
            updatePortfolioChart(period);
        });
    }
}

// Update Portfolio Chart based on time filter
function updatePortfolioChart(period) {
    if (!portfolioChart) return;
    
    let labels = [];
    let data = [];
    let baseValue = 10000;
    
    switch(period) {
        case '7d':
            // Last 7 days
            for (let i = 6; i >= 0; i--) {
                const date = new Date();
                date.setDate(date.getDate() - i);
                labels.push(date.toLocaleDateString('en-US', { weekday: 'short' }));
            }
            
            // Generate data with less volatility for short term
            let currentValue = baseValue;
            for (let i = 0; i <= 6; i++) {
                const dailyChange = (Math.random() * 4 - 1.5) / 100;
                currentValue *= (1 + dailyChange);
                data.push(currentValue);
            }
            break;
            
        case '30d':
            // Last 30 days (default)
            for (let i = 30; i >= 0; i--) {
                const date = new Date();
                date.setDate(date.getDate() - i);
                labels.push(date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' }));
            }
            
            currentValue = baseValue;
            for (let i = 0; i <= 30; i++) {
                const dailyChange = (Math.random() * 5 - 2) / 100;
                currentValue *= (1 + dailyChange);
                data.push(currentValue);
            }
            break;
            
        case '90d':
            // Last 90 days (monthly points)
            for (let i = 90; i >= 0; i -= 3) {
                const date = new Date();
                date.setDate(date.getDate() - i);
                labels.push(date.toLocaleDateString('en-US', { month: 'short', day: 'numeric' }));
            }
            
            currentValue = baseValue;
            for (let i = 0; i <= 90; i += 3) {
                const periodChange = (Math.random() * 8 - 3) / 100;
                currentValue *= (1 + periodChange);
                data.push(currentValue);
            }
            break;
            
        case '1y':
            // Last year (monthly points)
            for (let i = 12; i >= 0; i--) {
                const date = new Date();
                date.setMonth(date.getMonth() - i);
                labels.push(date.toLocaleDateString('en-US', { month: 'short', year: '2-digit' }));
            }
            
            currentValue = baseValue;
            for (let i = 0; i <= 12; i++) {
                const monthlyChange = (Math.random() * 10 - 3) / 100;
                currentValue *= (1 + monthlyChange);
                data.push(currentValue);
            }
            break;
    }
    
    // Update chart
    portfolioChart.data.labels = labels;
    portfolioChart.data.datasets[0].data = data;
    portfolioChart.update();
    
    // Update stats
    updateChartStats(data);
}

// Update chart statistics
function updateChartStats(data) {
    if (!data || data.length < 2) return;
    
    const startValue = data[0];
    const endValue = data[data.length - 1];
    const growth = ((endValue - startValue) / startValue) * 100;
    
    // Find best day
    let maxIncrease = 0;
    let bestDayValue = 0;
    
    for (let i = 1; i < data.length; i++) {
        const increase = data[i] - data[i-1];
        if (increase > maxIncrease) {
            maxIncrease = increase;
            bestDayValue = increase;
        }
    }
    
    // Calculate average daily change (skip first day as it's the starting point)
    let totalChange = 0;
    for (let i = 1; i < data.length; i++) {
        totalChange += data[i] - data[i-1];
    }
    const avgDaily = data.length > 1 ? totalChange / (data.length - 1) : 0;
    
    // Update UI
    const currentValueEl = document.querySelector('.chart-stat:nth-child(1) .stat-value');
    const totalGrowthEl = document.querySelector('.chart-stat:nth-child(2) .stat-value');
    const bestDayEl = document.querySelector('.chart-stat:nth-child(3) .stat-value');
    const avgDailyEl = document.querySelector('.chart-stat:nth-child(4) .stat-value');
    
    if (currentValueEl) currentValueEl.textContent = `$${endValue.toFixed(2)}`;
    if (totalGrowthEl) {
        totalGrowthEl.textContent = `${growth >= 0 ? '+' : ''}${growth.toFixed(1)}%`;
        totalGrowthEl.className = growth >= 0 ? 'stat-value positive' : 'stat-value negative';
    }
    if (bestDayEl) {
        bestDayEl.textContent = `$${bestDayValue.toFixed(2)}`;
        bestDayEl.className = bestDayValue >= 0 ? 'stat-value positive' : 'stat-value negative';
    }
    if (avgDailyEl) {
        avgDailyEl.textContent = `$${avgDaily.toFixed(2)}`;
        avgDailyEl.className = avgDaily >= 0 ? 'stat-value positive' : 'stat-value negative';
    }
}

// Security Status
function initSecurityStatus() {
    const securityItems = document.querySelectorAll('.security-item');
    
    securityItems.forEach(item => {
        const actionBtn = item.querySelector('.security-action');
        if (actionBtn) {
            actionBtn.addEventListener('click', function(e) {
                e.preventDefault();
                const securityTitle = item.querySelector('.security-title').textContent;
                
                if (window.CommonFunctions) {
                    window.CommonFunctions.showSuccessNotification(`Initiating ${securityTitle} verification process...`);
                }
                
                // Simulate verification process
                setTimeout(() => {
                    item.classList.remove('inactive');
                    item.classList.add('active');
                    
                    const icon = item.querySelector('.security-icon');
                    const status = item.querySelector('.security-status');
                    
                    icon.className = 'security-icon fas fa-check-circle';
                    status.textContent = 'Active';
                    
                    if (actionBtn) actionBtn.style.display = 'none';
                    
                    if (window.CommonFunctions) {
                        window.CommonFunctions.showSuccessNotification(`${securityTitle} verified successfully!`);
                    }
                }, 1500);
            });
        }
    });
}

// Market Data
function initMarketData() {
    const marketItems = document.querySelectorAll('.market-item');
    
    // Simulate live market updates
    setInterval(() => {
        marketItems.forEach(item => {
            const changeEl = item.querySelector('.market-change');
            if (changeEl && !changeEl.classList.contains('neutral')) {
                // Small random fluctuation
                const currentChange = parseFloat(changeEl.textContent);
                const fluctuation = (Math.random() * 0.5 - 0.25); // -0.25% to +0.25%
                const newChange = currentChange + fluctuation;
                
                // Update display
                const isPositive = newChange >= 0;
                changeEl.innerHTML = `<i class="fas fa-arrow-${isPositive ? 'up' : 'down'}"></i> ${Math.abs(newChange).toFixed(1)}%`;
                changeEl.className = isPositive ? 'market-change positive' : 'market-change negative';
                
                // Update price
                const priceEl = item.querySelector('.market-price');
                if (priceEl && !item.querySelector('.market-icon i').classList.contains('fa-coins')) {
                    const currentPrice = parseFloat(priceEl.textContent.replace('$', '').replace(',', ''));
                    const priceChange = currentPrice * (fluctuation / 100);
                    const newPrice = currentPrice + priceChange;
                    priceEl.textContent = `$${newPrice.toFixed(2)}`;
                }
            }
        });
    }, 10000); // Update every 10 seconds
}

// Transactions
function initTransactions() {
    const transactionItems = document.querySelectorAll('.transaction-item');
    
    transactionItems.forEach(item => {
        item.addEventListener('click', function() {
            const title = this.querySelector('.transaction-title').textContent;
            const amount = this.querySelector('.amount').textContent;
            const time = this.querySelector('.transaction-time').textContent;
            const status = this.querySelector('.status').textContent;
            
            showTransactionDetails({
                title,
                amount,
                time,
                status
            });
        });
    });
}

// Show Transaction Details
function showTransactionDetails(details) {
    // Create modal
    const overlay = document.createElement('div');
    overlay.className = 'modal-overlay';
    overlay.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.8);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: var(--z-modal);
        backdrop-filter: blur(5px);
    `;
    
    const modal = document.createElement('div');
    modal.className = 'modal-content';
    modal.innerHTML = `
        <div class="modal-header">
            <h3>Transaction Details</h3>
            <button class="modal-close">&times;</button>
        </div>
        <div class="modal-body">
            <div class="transaction-details-modal">
                <div class="detail-item">
                    <span class="detail-label">Transaction:</span>
                    <span class="detail-value">${details.title}</span>
                </div>
                <div class="detail-item">
                    <span class="detail-label">Amount:</span>
                    <span class="detail-value ${details.amount.startsWith('+') ? 'positive' : 'negative'}">${details.amount}</span>
                </div>
                <div class="detail-item">
                    <span class="detail-label">Time:</span>
                    <span class="detail-value">${details.time}</span>
                </div>
                <div class="detail-item">
                    <span class="detail-label">Status:</span>
                    <span class="detail-value status-${details.status.toLowerCase()}">${details.status}</span>
                </div>
                <div class="detail-item">
                    <span class="detail-label">Transaction ID:</span>
                    <span class="detail-value">TX${Math.random().toString(36).substr(2, 9).toUpperCase()}</span>
                </div>
                <div class="detail-item">
                    <span class="detail-label">Network:</span>
                    <span class="detail-value">TRC20</span>
                </div>
                <div class="detail-item">
                    <span class="detail-label">Confirmations:</span>
                    <span class="detail-value">12/12</span>
                </div>
            </div>
            
            <div class="modal-actions">
                <button type="button" class="btn-secondary close-modal">Close</button>
                <button type="button" class="btn-primary" onclick="window.print()">
                    <i class="fas fa-print"></i> Print Receipt
                </button>
            </div>
        </div>
    `;
    
    // Style modal
    modal.style.cssText = `
        background: var(--dark-light);
        border-radius: var(--border-radius-xl);
        padding: var(--spacing-2xl);
        width: 90%;
        max-width: 500px;
        border: 1px solid rgba(255, 255, 255, 0.1);
        box-shadow: var(--shadow-xl);
    `;
    
    // Add to page
    overlay.appendChild(modal);
    document.body.appendChild(overlay);
    
    // Close functionality
    const closeBtn = modal.querySelector('.modal-close');
    const closeModalBtn = modal.querySelector('.close-modal');
    
    const closeModal = () => {
        document.body.removeChild(overlay);
    };
    
    if (closeBtn) closeBtn.addEventListener('click', closeModal);
    if (closeModalBtn) closeModalBtn.addEventListener('click', closeModal);
    
    // Close when clicking outside
    overlay.addEventListener('click', function(e) {
        if (e.target === this) {
            closeModal();
        }
    });
    
    // Close with Escape key
    document.addEventListener('keydown', function(e) {
        if (e.key === 'Escape' && document.body.contains(overlay)) {
            closeModal();
        }
    });
}

// Quick Actions
function initQuickActions() {
    const quickActions = document.querySelectorAll('.quick-action');
    
    quickActions.forEach(action => {
        action.addEventListener('click', function(e) {
            const actionTitle = this.querySelector('.action-title').textContent;
            
            // Show loading animation
            const originalHTML = this.innerHTML;
            this.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Loading...';
            
            // Simulate loading
            setTimeout(() => {
                this.innerHTML = originalHTML;
                
                if (window.CommonFunctions) {
                    window.CommonFunctions.showSuccessNotification(`Redirecting to ${actionTitle}...`);
                }
                
                // In a real app, this would redirect to the appropriate page
                const href = this.getAttribute('href');
                if (href && href !== '#') {
                    window.location.href = href;
                }
            }, 1000);
        });
    });
    
    // Quick deposit button
    const quickDepositBtn = document.querySelector('.welcome-actions .btn-primary');
    if (quickDepositBtn) {
        quickDepositBtn.addEventListener('click', function() {
            window.location.href = 'deposit.html';
        });
    }
    
    // View analytics button
    const viewAnalyticsBtn = document.querySelector('.welcome-actions .btn-secondary');
    if (viewAnalyticsBtn) {
        viewAnalyticsBtn.addEventListener('click', function() {
            if (window.CommonFunctions) {
                window.CommonFunctions.showSuccessNotification('Opening advanced analytics dashboard...');
            }
        });
    }
}

// Animations
function initAnimations() {
    // Animate stat cards on scroll
    const statCards = document.querySelectorAll('.stat-card');
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };
    
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0)';
            }
        });
    }, observerOptions);
    
    statCards.forEach(card => {
        card.style.opacity = '0';
        card.style.transform = 'translateY(20px)';
        card.style.transition = 'opacity 0.5s ease, transform 0.5s ease';
        observer.observe(card);
    });
    
    // Animate welcome banner
    const welcomeBanner = document.querySelector('.welcome-banner');
    if (welcomeBanner) {
        welcomeBanner.style.opacity = '0';
        welcomeBanner.style.transform = 'translateY(-20px)';
        
        setTimeout(() => {
            welcomeBanner.style.transition = 'opacity 0.8s ease, transform 0.8s ease';
            welcomeBanner.style.opacity = '1';
            welcomeBanner.style.transform = 'translateY(0)';
        }, 300);
    }
    
    // Add hover effects to cards
    const cards = document.querySelectorAll('.card');
    cards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-5px)';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0)';
        });
    });
    
    // Animate progress bars
    const progressBars = document.querySelectorAll('.progress-fill');
    progressBars.forEach(bar => {
        const width = bar.style.width;
        bar.style.width = '0%';
        
        setTimeout(() => {
            bar.style.transition = 'width 1.5s ease';
            bar.style.width = width;
        }, 500);
    });
}

// Update dashboard stats periodically
function updateDashboardStats() {
    setInterval(() => {
        // Update total profit with small random increase
        const totalProfitEl = document.querySelector('.stat-card:nth-child(2) .stat-value');
        if (totalProfitEl) {
            const currentProfit = parseFloat(totalProfitEl.textContent.replace('$', '').replace(',', ''));
            const dailyIncrease = 45.20 + (Math.random() * 10 - 5); // Random variation
            const newProfit = currentProfit + dailyIncrease;
            totalProfitEl.textContent = `$${newProfit.toFixed(2)}`;
            
            // Update today's profit in breakdown
            const todayProfitEl = document.querySelector('.stat-card:nth-child(2) .stat-breakdown span:nth-child(1)');
            if (todayProfitEl) {
                todayProfitEl.textContent = `Today: +$${dailyIncrease.toFixed(2)}`;
            }
        }
        
        // Update available balance (simulate small changes)
        const balanceEl = document.querySelector('.stat-card:nth-child(1) .stat-value');
        if (balanceEl) {
            const currentBalance = parseFloat(balanceEl.textContent.replace('$', '').replace(',', ''));
            const change = (Math.random() * 10 - 5); // Small random change
            const newBalance = currentBalance + change;
            balanceEl.textContent = `$${newBalance.toFixed(2)}`;
        }
    }, 30000); // Update every 30 seconds
}

// Initialize real-time updates
setTimeout(() => {
    updateDashboardStats();
}, 5000);

// Export functions for debugging
window.Dashboard = {
    initDashboard,
    updatePortfolioChart,
    showTransactionDetails
};